# Controlled Human Malaria Infection (CHMI) ANOVA for area under the curve in
# four groups

# Author: Luc Coffeng
# Date created: 4 April 2017


# Prep session
  rm(list=ls())
  
  library(data.table)
  library(rstanarm)
  library(shinystan)
  library(bayesplot)
  library(ggplot2)
  library(plyr)
  library(coda)
  
  options(mc.cores = parallel::detectCores())
  
  base.dir <- "/Users/Luc/Documents/Research/Radboud UMC/Gametocytes/01_Analysis"
  code.dir <- file.path(base.dir,"01_Code")
  out.dir <- file.path(base.dir,"02_Output")
  data.file <- "/Users/Luc/Documents/Research/Data/Malaria/CHMI_gameto/Processed/layout Luc AUC vs peak gct concentration.csv"
  
  
# Load functions
  setwd(code.dir)
  source("CHMI_anova_AUC_gameto_functions.r")
  
  
# Load and prep data
  maldata <- fread(data.file)
  
  maldata[Group %in% 1:2, Treat1 := "SP_low"]
  maldata[Group %in% 3:4, Treat1 := "Pip_low"]
  maldata[Group %in% c(1,4), Treat2 := "SP_high"]
  maldata[Group %in% 2:3, Treat2 := "Pip_high"]
  
  maldata[, Group := as.factor(Group)]
  
  
# Fit ANOVA model for AUC gct
  fit.anova.classic.gct <- aov(log10_AUC_gct_dlimit5 ~ Treat1 * Treat2, data = maldata)
  
  fit.lm.bayesian.gct <- stan_lm(log10_AUC_gct_dlimit5 ~ Group,
                                 data = maldata,
                                 # prior_intercept = normal(location = 2.5, scale = 2, autoscale = FALSE),
                                 prior_intercept = NULL,
                                 # prior = R2(location = 0.25),
                                 prior = NULL,
                                 chains = parallel::detectCores(),
                                 cores = parallel::detectCores(),
                                 seed = 12345,
                                 adapt_delta = 0.999,
                                 iter = 10000)
  summary(fit.anova.classic.gct)
  summary(fit.lm.bayesian.gct, digits = 3)
  
  pdf(file = "AUC absolute per group_gameto.pdf", width = 6, height = 5)
  
    for(i in c("red", "yellow", "blue", "green" #, "brightblue", "pink", "purple",
              #"teal", "viridis", "viridisA", "viridisB", "viridisC"
              )) {
      color_scheme_set(scheme = i)
      plot.mean.per.group(fit = fit.lm.bayesian.gct,
                          unit = "Gametocyte concentration AUC",
                          xlimits = log10(c(1e1, 3e4)),
                          # breaks = function(x) {
                          #   lo <- floor(x[1])
                          #   hi <- floor(x[2])
                          #   c(sapply(lo:hi, function(y) y + log10(c(1, 2, 5))), hi + 1)
                          # },
                          # breaks = 1:4,
                          breaks = log10(c(1:9 %*% t(10^(1:3)), 1e4, 2e4, 3e4)),
                          labels = function(x) {
                            ifelse(x %in% log10(10^(1:4)),
                                   format(10^x, big.mark = ",", scientific = FALSE),
                                   "")})
    }
  
  dev.off()
  
  pdf(file = "AUC absolute per group_gameto_simple.pdf", width = 9, height = 5)
  for(i in c("red", "yellow", "blue", "green" #, "brightblue", "pink", "purple",
             #"teal", "viridis", "viridisA", "viridisB", "viridisC"
  )) {
    color_scheme_set(scheme = i)
    plot.mean.per.group(fit = fit.lm.bayesian.gct,
                        unit = "Gametocyte concentration AUC",
                        xlimits = log10(c(1e1, 2e5)),
                        # breaks = 2:10 / 2,
                        # labels = 2:10 / 2)
                        breaks = log10(c(1:9 %*% t(10^(1:5)))),
                        labels = function(x) {
                          ifelse(x %in% log10(10^(1:5)),
                                 format(10^x, big.mark = ",", scientific = FALSE),
                                 "")})
  }
  
  dev.off()
  
  plot.results(fit = fit.lm.bayesian.gct,
               pdf.name = "AUC difference per group_gameto.pdf",
               unit = "log-10 AUC gct")
  
  
  
# Fit ANOVA model for AUC asex
  fit.anova.classic.asex <- aov(log10_AUC_asex ~ Treat1 * Treat2, data = maldata)
  
  fit.lm.bayesian.asex <- stan_lm(log10_AUC_asex ~ Group,
                                 data = maldata,
                                 # prior_intercept = normal(location = 2.5, scale = 2, autoscale = FALSE),
                                 prior_intercept = NULL,
                                 # prior = R2(location = 0.25),
                                 prior = NULL,
                                 chains = parallel::detectCores(),
                                 cores = parallel::detectCores(),
                                 seed = 12345,
                                 adapt_delta = 0.999,
                                 iter = 10000)
  summary(fit.anova.classic.asex)
  summary(fit.lm.bayesian.asex, digits = 3)
  
  pdf(file = "AUC absolute per group_asex.pdf", width = 6, height = 5)
    
  for(i in c("red", "yellow", "blue", "green" #, "brightblue", "pink", "purple",
             #"teal", "viridis", "viridisA", "viridisB", "viridisC"
             )) {
      color_scheme_set(scheme = i)
      plot.mean.per.group(fit = fit.lm.bayesian.asex,
                          unit = "Asexual parasite concentration AUC",
                          xlimits = log10(c(2e3, 5e4)))
    }
    
  dev.off()
  
  plot.results(fit = fit.lm.bayesian.asex,
               pdf.name = "AUC difference per group_asex.pdf",
               unit = "log-10 AUC asex")
  
# Fit ANOVA model for first peak asex
  fit.anova.classic.asex.peak1 <- aov(Log10_First_peak_asex ~ Treat1 * Treat2, data = maldata)
  
  fit.lm.bayesian.asex.peak1 <- stan_lm(Log10_First_peak_asex ~ Group,
                                        data = maldata,
                                        # prior_intercept = normal(location = 2.5, scale = 2, autoscale = FALSE),
                                        prior_intercept = NULL,
                                        # prior = R2(location = 0.25),
                                        prior = NULL,
                                        chains = parallel::detectCores(),
                                        cores = parallel::detectCores(),
                                        seed = 12345,
                                        adapt_delta = 0.999,
                                        iter = 10000)
  summary(fit.anova.classic.asex.peak1)
  summary(fit.lm.bayesian.asex.peak1, digits = 3)
  
  plot.results(fit = fit.lm.bayesian.asex.peak1,
               pdf.name = "First peak difference per group_asex.pdf",
               unit = "first peak asex")
 
  
# Fit ANOVA model for AUC gct, corrected for AUC asex
  fit.lm.bayesian.gct.corrected <- stan_lm(log10_AUC_gct_dlimit5 ~ Group + log10_AUC_asex,
                                           data = maldata,
                                           # prior_intercept = normal(location = 2.5, scale = 2, autoscale = FALSE),
                                           prior_intercept = NULL,
                                           # prior = R2(location = 0.25),
                                           prior = NULL,
                                           chains = parallel::detectCores(),
                                           cores = parallel::detectCores(),
                                           seed = 12345,
                                           adapt_delta = 0.999,
                                           iter = 10000)
  summary(fit.lm.bayesian.gct.corrected, digits = 3)
  
  plot.results(fit = fit.lm.bayesian.gct.corrected,
               pdf.name = "AUC difference per group_gameto_corrected for asex AUC.pdf",
               unit = "log-10 AUC gct (corrected for AUC asex)",
               absolute = FALSE)  # don't print absolute levels as the code does not account for an extra predictor
   
# Correlations
  maldata[, cor.test(log10_AUC_asex, log10_AUC_gct_dlimit5, method = "pearson")]  # signif.
  maldata[, cor.test(Log10_First_peak_asex, Log10_First_peak_gct, method = "pearson")]  # n.s.
  maldata[, cor.test(Log10_First_peak_asex, Log10_Peak_gct, method = "pearson")]  # n.s.
  maldata[, cor.test(Log10_First_peak_asex, log10_AUC_gct_dlimit5, method = "pearson")]  # n.s.
  maldata[, cor.test(Log10_First_peak_asex, log10_AUC_asex, method = "pearson")]  # n.s.
  maldata[, sd(Log10_First_peak_asex) / mean(Log10_First_peak_asex)]  # relatively high variation
  maldata[, sd(log10_AUC_asex) / mean(log10_AUC_asex)]  # relatively low variation
  
### END OF CODE
  
  
  